<?php
declare(strict_types=1);
session_name(app_config()['app']['session_name'] ?? 'offline_stock_sess');
session_start();

require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/flash.php';
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/csrf.php';

require_login();
$pdo = db();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_verify();
  if (!isset($_FILES['csv']) || $_FILES['csv']['error'] !== UPLOAD_ERR_OK) {
    flash_set('error','CSV upload failed.');
    redirect('/public/import_products.php');
  }

  $tmp = $_FILES['csv']['tmp_name'];
  $fh = fopen($tmp, 'r');
  if (!$fh) { flash_set('error','Cannot read CSV.'); redirect('/public/import_products.php'); }

  $header = fgetcsv($fh);
  if (!$header) { flash_set('error','Empty CSV.'); redirect('/public/import_products.php'); }

  // Expected: sku,name,category,price,stock,reorder_level
  $map = array_flip(array_map('strtolower', $header));

  $required = ['sku','name','category','price','stock','reorder_level'];
  foreach ($required as $col) {
    if (!array_key_exists($col, $map)) {
      flash_set('error',"Missing column: {$col}");
      redirect('/public/import_products.php');
    }
  }

  $created = 0; $updated = 0;

  while (($row = fgetcsv($fh)) !== false) {
    $sku = trim($row[$map['sku']] ?? '');
    $name = trim($row[$map['name']] ?? '');
    $catName = trim($row[$map['category']] ?? '');
    $price = (float)($row[$map['price']] ?? 0);
    $stock = (int)($row[$map['stock']] ?? 0);
    $reorder = trim((string)($row[$map['reorder_level']] ?? ''));
    $reorder = ($reorder === '' ? null : (int)$reorder);

    if ($sku === '' || $name === '') continue;

    // category create/get
    $catId = null;
    if ($catName !== '') {
      $st = $pdo->prepare("SELECT id FROM categories WHERE name=? LIMIT 1");
      $st->execute([$catName]);
      $cat = $st->fetch();
      if ($cat) $catId = (int)$cat['id'];
      else {
        $pdo->prepare("INSERT INTO categories(name, created_at) VALUES(?,?)")->execute([$catName, now()]);
        $catId = (int)$pdo->lastInsertId();
      }
    }

    // upsert by sku
    $st = $pdo->prepare("SELECT id, sell_price, stock_qty FROM products WHERE sku=? LIMIT 1");
    $st->execute([$sku]);
    $p = $st->fetch();

    if (!$p) {
      $pdo->prepare("INSERT INTO products(sku,name,category_id,cost_price,sell_price,stock_qty,reorder_level,is_active,created_at,updated_at)
                     VALUES(?,?,?,?,?,?,?,?,?,?)")
          ->execute([$sku,$name,$catId,0,$price,$stock,$reorder,1,now(),now()]);
      $pid = (int)$pdo->lastInsertId();
      if ($stock !== 0) {
        $pdo->prepare("INSERT INTO stock_movements(product_id,type,qty,note,user_id,created_at) VALUES(?,?,?,?,?,?)")
            ->execute([$pid,'ADJUST',$stock,'CSV import initial stock', current_user()['id'], now()]);
      }
      $created++;
    } else {
      $pid = (int)$p['id'];
      $pdo->beginTransaction();
      try {
        // update name/category/reorder + price + stock with history/movements
        $pdo->prepare("UPDATE products SET name=?, category_id=?, reorder_level=?, updated_at=? WHERE id=?")
            ->execute([$name,$catId,$reorder,now(),$pid]);

        if ((float)$p['sell_price'] !== $price) {
          $pdo->prepare("UPDATE products SET sell_price=?, updated_at=? WHERE id=?")->execute([$price, now(), $pid]);
          $pdo->prepare("INSERT INTO price_history(product_id, old_price, new_price, user_id, created_at) VALUES(?,?,?,?,?)")
              ->execute([$pid, (float)$p['sell_price'], $price, current_user()['id'], now()]);
        }

        if ((int)$p['stock_qty'] !== $stock) {
          $delta = $stock - (int)$p['stock_qty'];
          $pdo->prepare("UPDATE products SET stock_qty=?, updated_at=? WHERE id=?")->execute([$stock, now(), $pid]);
          $pdo->prepare("INSERT INTO stock_movements(product_id,type,qty,note,user_id,created_at) VALUES(?,?,?,?,?,?)")
              ->execute([$pid,'ADJUST',$delta,'CSV import update', current_user()['id'], now()]);
        }

        $pdo->commit();
        $updated++;
      } catch (Throwable $e) {
        $pdo->rollBack();
      }
    }
  }

  fclose($fh);
  flash_set('success', "Import done. Created: {$created}, Updated: {$updated}.");
  redirect('/public/products.php');
}

$title = 'Import Products';
require_once __DIR__ . '/../app/header.php';
?>
<h1>Import Products (CSV)</h1>

<div class="panel">
  <p class="muted small">
    Upload CSV with columns: <b>sku,name,category,price,stock,reorder_level</b>.
    If category doesn't exist, it will be created.
  </p>

  <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
    <div class="field">
      <label>CSV File</label>
      <input type="file" name="csv" accept=".csv,text/csv" required>
    </div>
    <button type="submit">Import</button>
  </form>

  <hr>
  <p class="muted small">Tip: Export CSV from Products page first, edit offline in Excel, then import again.</p>
</div>

<?php require_once __DIR__ . '/../app/footer.php'; ?>
