<?php
declare(strict_types=1);
session_name(app_config()['app']['session_name'] ?? 'offline_stock_sess');
session_start();

require_once __DIR__ . '/../app/db.php';
require_once __DIR__ . '/../app/helpers.php';
require_once __DIR__ . '/../app/flash.php';
require_once __DIR__ . '/../app/auth.php';
require_once __DIR__ . '/../app/csrf.php';

require_login();

$pdo = db();

$q = trim($_GET['q'] ?? '');
$category_id = (int)($_GET['category_id'] ?? 0);

if (($_GET['export'] ?? '') === 'csv') {
  $sql = "SELECT p.sku,p.name,c.name AS category,p.sell_price AS price,p.stock_qty AS stock,p.reorder_level
          FROM products p LEFT JOIN categories c ON c.id=p.category_id WHERE 1=1";
  $params = [];
  if ($q !== '') { $sql .= " AND (p.sku LIKE ? OR p.name LIKE ?)"; $params[] = "%$q%"; $params[] = "%$q%"; }
  if ($category_id > 0) { $sql .= " AND p.category_id=?"; $params[] = $category_id; }
  $sql .= " ORDER BY p.id DESC";
  $stmt = $pdo->prepare($sql);
  $stmt->execute($params);
  $rows = $stmt->fetchAll();

  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="products_export.csv"');
  $out = fopen('php://output', 'w');
  fputcsv($out, ['sku','name','category','price','stock','reorder_level']);
  foreach ($rows as $r) {
    fputcsv($out, [$r['sku'],$r['name'],$r['category'],$r['price'],$r['stock'],$r['reorder_level']]);
  }
  fclose($out);
  exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_verify();
  $action = $_POST['action'] ?? '';
  if ($action === 'create') {
    $sku = trim($_POST['sku'] ?? '');
    $name = trim($_POST['name'] ?? '');
    $cat = (int)($_POST['category_id'] ?? 0);
    $cost = (float)($_POST['cost_price'] ?? 0);
    $sell = (float)($_POST['sell_price'] ?? 0);
    $stock = (int)($_POST['stock_qty'] ?? 0);
    $reorder = ($_POST['reorder_level'] ?? '') === '' ? null : (int)$_POST['reorder_level'];

    if ($sku === '' || $name === '') { flash_set('error','SKU and Name are required.'); redirect('/public/products.php'); }

    $stmt = $pdo->prepare("INSERT INTO products(sku,name,category_id,cost_price,sell_price,stock_qty,reorder_level,is_active,created_at,updated_at)
                           VALUES(?,?,?,?,?,?,?,?,?,?)");
    $stmt->execute([$sku,$name,$cat?:null,$cost,$sell,$stock,$reorder,1,now(),now()]);

    // movement log (initial stock as ADJUST if >0)
    $pid = (int)$pdo->lastInsertId();
    if ($stock !== 0) {
      $pdo->prepare("INSERT INTO stock_movements(product_id,type,qty,note,user_id,created_at) VALUES(?,?,?,?,?,?)")
          ->execute([$pid,'ADJUST',$stock,'Initial stock', current_user()['id'], now()]);
    }

    flash_set('success','Product created.');
    redirect('/public/products.php');
  }
  if ($action === 'delete') {
    $id = (int)($_POST['id'] ?? 0);
    $pdo->prepare("DELETE FROM stock_movements WHERE product_id=?")->execute([$id]);
    $pdo->prepare("DELETE FROM products WHERE id=?")->execute([$id]);
    flash_set('success','Product deleted.');
    redirect('/public/products.php');
  }
}

$cats = $pdo->query("SELECT id,name FROM categories ORDER BY name ASC")->fetchAll();

$sql = "SELECT p.*, c.name AS category_name
        FROM products p LEFT JOIN categories c ON c.id=p.category_id
        WHERE 1=1";
$params = [];
if ($q !== '') { $sql .= " AND (p.sku LIKE ? OR p.name LIKE ?)"; $params[] = "%$q%"; $params[] = "%$q%"; }
if ($category_id > 0) { $sql .= " AND p.category_id=?"; $params[] = $category_id; }
$sql .= " ORDER BY p.id DESC LIMIT 300";
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$products = $stmt->fetchAll();

$title = 'Products';
require_once __DIR__ . '/../app/header.php';
?>
<h1>Products</h1>

<div class="panel">
  <form method="get" class="row" style="flex-wrap:wrap;">
    <div class="field" style="margin:0;min-width:260px;flex:1;">
      <label class="small muted">Search</label>
      <input name="q" value="<?= e($q) ?>" placeholder="Search SKU / Name">
    </div>
    <div class="field" style="margin:0;min-width:220px;">
      <label class="small muted">Category</label>
      <select name="category_id">
        <option value="0">All</option>
        <?php foreach($cats as $c): ?>
          <option value="<?= e((string)$c['id']) ?>" <?= $category_id===(int)$c['id']?'selected':'' ?>><?= e($c['name']) ?></option>
        <?php endforeach; ?>
      </select>
    </div>
    <div class="field" style="margin:0;align-self:end;">
      <button type="submit">Filter</button>
      <a class="btn" href="<?= e(url('/public/products.php?export=csv&category_id='.$category_id.'&q='.urlencode($q))) ?>">Export CSV</a>
      <a class="btn" href="<?= e(url('/public/import_products.php')) ?>">Import CSV</a>
    </div>
  </form>
</div>

<div class="grid grid-2" style="margin-top:14px;">
  <div class="panel">
    <h2>Add product</h2>
    <form method="post">
      <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
      <input type="hidden" name="action" value="create">

      <div class="grid grid-2">
        <div class="field">
          <label>SKU</label>
          <input name="sku" placeholder="e.g. PNJ-001" required>
        </div>
        <div class="field">
          <label>Name</label>
          <input name="name" placeholder="Product name" required>
        </div>
      </div>

      <div class="grid grid-3">
        <div class="field">
          <label>Category</label>
          <select name="category_id">
            <option value="0">(None)</option>
            <?php foreach($cats as $c): ?>
              <option value="<?= e((string)$c['id']) ?>"><?= e($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="field">
          <label>Cost Price</label>
          <input name="cost_price" type="number" step="0.01" value="0">
        </div>
        <div class="field">
          <label>Selling Price</label>
          <input name="sell_price" type="number" step="0.01" value="0">
        </div>
      </div>

      <div class="grid grid-3">
        <div class="field">
          <label>Initial Stock</label>
          <input name="stock_qty" type="number" value="0">
        </div>
        <div class="field">
          <label>Reorder Level (Low stock)</label>
          <input name="reorder_level" type="number" placeholder="e.g. 5">
        </div>
        <div class="field">
          <label>Status</label>
          <input value="Active" disabled>
        </div>
      </div>

      <button type="submit">Create</button>
    </form>
  </div>

  <div class="panel">
    <h2>Latest products (max 300)</h2>
    <p class="muted small">For fast bulk updates, use <b>Quick Update</b>.</p>
    <div style="overflow:auto;">
      <table class="table">
        <thead>
          <tr>
            <th>SKU</th><th>Name</th><th>Category</th><th>Price</th><th>Stock</th><th>Low?</th><th>Action</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach($products as $p): 
            $low = ($p['reorder_level'] !== null && (int)$p['stock_qty'] <= (int)$p['reorder_level']);
          ?>
            <tr>
              <td><?= e($p['sku']) ?></td>
              <td><?= e($p['name']) ?></td>
              <td><?= e($p['category_name'] ?? '-') ?></td>
              <td><?= e(number_format((float)$p['sell_price'],2)) ?></td>
              <td><?= e((string)$p['stock_qty']) ?></td>
              <td><?= $low ? '<span class="badge bad">LOW</span>' : '<span class="badge ok">OK</span>' ?></td>
              <td>
                <form method="post" onsubmit="return confirm('Delete this product?');">
                  <input type="hidden" name="csrf_token" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="action" value="delete">
                  <input type="hidden" name="id" value="<?= e((string)$p['id']) ?>">
                  <button type="submit">Delete</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php require_once __DIR__ . '/../app/footer.php'; ?>
